/*

	Project: Unequal Growth
	
	Purpose: Create portfolio composition figure
	
	Data: IMA Table S.3, NIPA Table 2.1, DINAs, DFAs
	
*/

clear all
cls

cd /Users/nilshaakonlehr/Dropbox/data_nils/Spring2020

global data_ima 	HH_Return_BEA/raw_data
global data_dina 	IRS_SZ_Figure/raw_data/PSZ2017Dinafiles
global data_dfa 	DFA/raw_data

********************************************************************************
* ------------------------------------------------------------------------------
* - DINA data -
* ------------------------------------------------------------------------------
********************************************************************************


* - Select variables -
local base id peinc hw* fk*

* - Loop specification -
local start_year 1966
local end_year 2012

qui{
foreach weight in dweght {
	foreach income in   peinc {
		di "Current variable: `income'"
		
		forvalues year = `start_year'/`end_year' {

			di "  "
			di "Current Year: " `year'

			* - Load data -
			use `weight' `base'  using $data_dina/usdina`year'.dta, clear
			
			* - Collapse data by tax id -
			gcollapse(sum) `weight' (mean) peinc  hw* fk*, by(id)
			
			* - Transform values of dweght to more standard range -
			qui replace `weight' = `weight'/10^5
		
			* - Generate percentiles/ permilles - 
			xtile decile = `income' [aw = `weight'], n(10)
			xtile percentile = `income' [aw = `weight'], n(100)
			xtile permille = `income' [aw = `weight'], n(1000)
			
			foreach var of varlist hw*  fk* `weight' {
				gen `var'_dec = `var' if decile == 10
				gen `var'_cen = `var' if percentile == 100
				gen `var'_mil = `var' if permille == 1000
			}
		
			* - Collapse variables -
			gcollapse(rawsum)  `weight' (sum) hw* fk* [iw = `weight']
		
			* - Generate observation year -
			gen year = `year'
		
			* - Store data -
			if (year == `start_year') {
				qui save clean_data/temp.dta,replace
			}
			else {
				qui append using clean_data/temp.dta			
				qui save clean_data/temp.dta,replace
			}
		}
	}
}
}

foreach var of varlist hw* fk*{
	replace `var' = `var'/10^12
}

* - Generate variables ---------------------------------------------------------

* - Wealth aggregates -
gen dina_w_equ = hwequ + hwbus
gen dina_w_fass = hwequ + hwfix + hwbus + hwpen
gen dina_w_ass = dina_w_fass + hwhou
gen dina_w_pen = hwpen

gen dina_w_fass_net = dina_w_fass + hwdeb
gen dina_w_ass_net = dina_w_ass + hwdeb

* - Income aggregates -
gen dina_i_equ = fkequ + fkbus
gen dina_i_fass = fkequ + fkfix + fkbus + fkpen
gen dina_i_ass = dina_i_fass + fkhou
gen dina_i_pen = fkpen

gen dina_i_fass_net = dina_i_fass + fkdeb
gen dina_i_ass_net = dina_i_ass + fkdeb


* - Disaggregated measures -
foreach stub in dec cen mil {
	* - Wealth -
	gen dina_w_equ_`stub' 		= hwequ_`stub' + hwbus_`stub'
	gen dina_w_fass_`stub' 	= dina_w_equ_`stub' + hwfix_`stub' + hwpen_`stub'
	gen dina_w_ass_`stub' 		= dina_w_fass_`stub' + hwhou_`stub'
	gen dina_w_pen_`stub' 		= hwpen_`stub'
	
	gen dina_w_fass_net_`stub' 	= dina_w_fass_`stub' + hwdeb_`stub'
	gen dina_w_ass_net_`stub' 		= dina_w_ass_`stub' + hwdeb_`stub'
	
	* - Income -
	gen dina_i_equ_`stub' 		= fkequ_`stub' + fkbus_`stub'
	gen dina_i_fass_`stub' 	= dina_i_equ_`stub' + fkfix_`stub' + fkpen_`stub'
	gen dina_i_ass_`stub' 		= dina_i_fass_`stub' + fkhou_`stub'
	gen dina_i_pen_`stub' 		= fkpen_`stub'
	
	gen dina_i_fass_net_`stub' 	= dina_i_fass_`stub' + fkdeb_`stub'
	gen dina_i_ass_net_`stub' 		= dina_i_ass_`stub' + fkdeb_`stub'
}

drop hw* dweght* fk*

* - Asset shares -
gen dina_sh_equ_fass = dina_w_equ/dina_w_fass_net*100
gen dina_sh_equ_fass_nopen = dina_w_equ/(dina_w_fass_net - dina_w_pen)*100
gen dina_sh_pen_fass = dina_w_pen/dina_w_fass_net*100

gen dina_sh_equ_ass = dina_w_equ/dina_w_ass_net*100
gen dina_sh_equ_ass_nopen = dina_w_equ/(dina_w_ass_net-dina_w_pen)*100

foreach stub in dec cen mil {
	gen dina_sh_equ_fass_`stub' = (dina_w_equ_`stub'/dina_w_fass_net_`stub')*100
	gen dina_sh_equ_fass_nopen_`stub' = dina_w_equ_`stub'/(dina_w_fass_net_`stub' - dina_w_pen_`stub')*100
	gen dina_sh_pen_fass_`stub' = dina_w_pen_`stub'/dina_w_fass_net_`stub'*100
	
	gen dina_sh_equ_ass_`stub' = (dina_w_equ_`stub'/dina_w_ass_net_`stub')*100
	gen dina_sh_equ_ass_nopen_`stub' = dina_w_equ_`stub'/(dina_w_ass_net_`stub'-dina_w_pen_`stub')*100
}

* - Income to wealth ratios -
gen dina_iwr_fass = dina_i_fass_net/dina_w_fass_net*100
gen dina_iwr_fass_nopen = (dina_i_fass_net - dina_i_pen)/(dina_w_fass_net - dina_w_pen) *100

gen dina_iwr_ass = dina_i_ass_net/dina_w_ass_net*100
gen dina_iwr_ass_nopen = (dina_i_ass_net - dina_i_pen)/(dina_w_ass_net - dina_w_pen)*100

foreach stub in dec cen mil {
	gen dina_iwr_fass_`stub' = (dina_i_fass_net_`stub'/dina_w_fass_net_`stub')*100
	gen dina_iwr_fass_nopen_`stub' = (dina_i_fass_net_`stub' - dina_i_pen_`stub')/(dina_w_fass_net_`stub' - dina_w_pen_`stub') *100
	
	gen dina_iwr_ass_`stub' = (dina_i_ass_net_`stub'/dina_w_ass_net_`stub')*100
	gen dina_iwr_ass_nopen_`stub' = (dina_i_ass_net_`stub' - dina_i_pen_`stub')/(dina_w_ass_net_`stub' - dina_w_pen_`stub')*100
}

* - Counter-factual shares holding constant pension wealth share -
gen nonpension_share = (dina_w_fass_net - dina_w_pen)/dina_w_fass_net
qui sum nonpension_share if year == 1980
replace nonpension_share = `r(mean)'
gen dina_sh_equ_fass_copen = dina_w_equ/(dina_w_fass_net - dina_w_pen)* nonpension_share * 100
drop nonpension_share

foreach stub in dec cen mil {
	gen nonpension_share = (dina_w_fass_net_`stub' - dina_w_pen_`stub')/dina_w_fass_net_`stub'
	qui sum nonpension_share if year == 1980
	replace nonpension_share = `r(mean)'
	gen dina_sh_equ_fass_copen_`stub' = dina_w_equ_`stub'/(dina_w_fass_net_`stub' - dina_w_pen_`stub')* nonpension_share * 100
	drop nonpension_share

}

keeporder year dina_iwr* dina_sh_*

save clean_data/dina_temp.dta,replace


********************************************************************************
* ------------------------------------------------------------------------------
* - DINA data for age groups -
* ------------------------------------------------------------------------------
********************************************************************************


qui{
foreach age in 20 45 65 {
	* - Select variables -
	local base id age peinc hw* fk*

	* - Loop specification -
	local start_year 1979
	local end_year 2012

	qui{
	foreach weight in dweght {
		foreach income in   peinc {
			di "Current variable: `income'"
			
			forvalues year = `start_year'/`end_year' {

				di "  "
				di "Current Year: " `year'

				* - Load data -
				use `weight' `base'  using $data_dina/usdina`year'.dta if age == `age', clear
				
				* - Collapse data by tax id -
				gcollapse(sum) `weight' (mean) peinc  hw* fk*, by(id)
				
				* - Transform values of dweght to more standard range -
				qui replace `weight' = `weight'/10^5
			
				* - Generate percentiles/ permilles - 
				xtile decile = `income' [aw = `weight'], n(10)
				xtile percentile = `income' [aw = `weight'], n(100)
				xtile permille = `income' [aw = `weight'], n(1000)
				
				foreach var of varlist hw*  fk* `weight' {
					gen `var'_dec = `var' if decile == 10
					gen `var'_cen = `var' if percentile == 100
					gen `var'_mil = `var' if permille == 1000
				}
			
				* - Collapse variables -
				gcollapse(rawsum)  `weight' (sum) hw* fk* [iw = `weight']
			
				* - Generate observation year -
				gen year = `year'
			
				* - Store data -
				if (year == `start_year') {
					qui save clean_data/temp.dta,replace
				}
				else {
					qui append using clean_data/temp.dta			
					qui save clean_data/temp.dta,replace
				}
			}
		}
	}
	}

	foreach var of varlist hw* fk*{
		replace `var' = `var'/10^12
	}

	* - Generate variables ---------------------------------------------------------

	* - Wealth aggregates -
	gen dina_w_equ = hwequ + hwbus
	gen dina_w_fass = hwequ + hwfix + hwbus + hwpen
	gen dina_w_ass = dina_w_fass + hwhou
	gen dina_w_pen = hwpen

	gen dina_w_fass_net = dina_w_fass + hwdeb
	gen dina_w_ass_net = dina_w_ass + hwdeb

	* - Income aggregates -
	gen dina_i_equ = fkequ + fkbus
	gen dina_i_fass = fkequ + fkfix + fkbus + fkpen
	gen dina_i_ass = dina_i_fass + fkhou
	gen dina_i_pen = fkpen

	gen dina_i_fass_net = dina_i_fass + fkdeb
	gen dina_i_ass_net = dina_i_ass + fkdeb


	* - Disaggregated measures -
	foreach stub in dec cen mil {
		* - Wealth -
		gen dina_w_equ_`stub' 		= hwequ_`stub' + hwbus_`stub'
		gen dina_w_fass_`stub' 	= dina_w_equ_`stub' + hwfix_`stub' + hwpen_`stub'
		gen dina_w_ass_`stub' 		= dina_w_fass_`stub' + hwhou_`stub'
		gen dina_w_pen_`stub' 		= hwpen_`stub'
		
		gen dina_w_fass_net_`stub' 	= dina_w_fass_`stub' + hwdeb_`stub'
		gen dina_w_ass_net_`stub' 		= dina_w_ass_`stub' + hwdeb_`stub'
		
		* - Income -
		gen dina_i_equ_`stub' 		= fkequ_`stub' + fkbus_`stub'
		gen dina_i_fass_`stub' 	= dina_i_equ_`stub' + fkfix_`stub' + fkpen_`stub'
		gen dina_i_ass_`stub' 		= dina_i_fass_`stub' + fkhou_`stub'
		gen dina_i_pen_`stub' 		= fkpen_`stub'
		
		gen dina_i_fass_net_`stub' 	= dina_i_fass_`stub' + fkdeb_`stub'
		gen dina_i_ass_net_`stub' 		= dina_i_ass_`stub' + fkdeb_`stub'
	}

	drop hw* dweght* fk*

	* - Asset shares -
	gen dina_sh_equ_fass = dina_w_equ/dina_w_fass_net*100
	gen dina_sh_equ_fass_nopen = dina_w_equ/(dina_w_fass_net - dina_w_pen)*100
	gen dina_sh_pen_fass = dina_w_pen/dina_w_fass_net*100

	gen dina_sh_equ_ass = dina_w_equ/dina_w_ass_net*100
	gen dina_sh_equ_ass_nopen = dina_w_equ/(dina_w_ass_net-dina_w_pen)*100

	foreach stub in dec cen mil {
		gen dina_sh_equ_fass_`stub' = (dina_w_equ_`stub'/dina_w_fass_net_`stub')*100
		gen dina_sh_equ_fass_nopen_`stub' = dina_w_equ_`stub'/(dina_w_fass_net_`stub' - dina_w_pen_`stub')*100
		gen dina_sh_pen_fass_`stub' = dina_w_pen_`stub'/dina_w_fass_net_`stub'*100
		
		gen dina_sh_equ_ass_`stub' = (dina_w_equ_`stub'/dina_w_ass_net_`stub')*100
		gen dina_sh_equ_ass_nopen_`stub' = dina_w_equ_`stub'/(dina_w_ass_net_`stub'-dina_w_pen_`stub')*100
	}

	* - Income to wealth ratios -
	gen dina_iwr_fass = dina_i_fass_net/dina_w_fass_net*100
	gen dina_iwr_fass_nopen = (dina_i_fass_net - dina_i_pen)/(dina_w_fass_net - dina_w_pen) *100

	gen dina_iwr_ass = dina_i_ass_net/dina_w_ass_net*100
	gen dina_iwr_ass_nopen = (dina_i_ass_net - dina_i_pen)/(dina_w_ass_net - dina_w_pen)*100

	foreach stub in dec cen mil {
		gen dina_iwr_fass_`stub' = (dina_i_fass_net_`stub'/dina_w_fass_net_`stub')*100
		gen dina_iwr_fass_nopen_`stub' = (dina_i_fass_net_`stub' - dina_i_pen_`stub')/(dina_w_fass_net_`stub' - dina_w_pen_`stub') *100
		
		gen dina_iwr_ass_`stub' = (dina_i_ass_net_`stub'/dina_w_ass_net_`stub')*100
		gen dina_iwr_ass_nopen_`stub' = (dina_i_ass_net_`stub' - dina_i_pen_`stub')/(dina_w_ass_net_`stub' - dina_w_pen_`stub')*100
	}

	* - Counter-factual shares holding constant pension wealth share -
	gen nonpension_share = (dina_w_fass_net - dina_w_pen)/dina_w_fass_net
	qui sum nonpension_share if year == 1980
	replace nonpension_share = `r(mean)'
	gen dina_sh_equ_fass_copen = dina_w_equ/(dina_w_fass_net - dina_w_pen)* nonpension_share *100
	drop nonpension_share

	foreach stub in dec cen mil {
		gen nonpension_share = (dina_w_fass_net_`stub' - dina_w_pen_`stub')/dina_w_fass_net_`stub'
		qui sum nonpension_share if year == 1980
		replace nonpension_share = `r(mean)'
		gen dina_sh_equ_fass_copen_`stub' = dina_w_equ_`stub'/(dina_w_fass_net_`stub' - dina_w_pen_`stub')* nonpension_share *100
		drop nonpension_share

	}
	keeporder year dina_iwr* dina_sh_*

	foreach var of varlist dina_iwr* dina_sh* {
		rename `var' `var'_a`age'
	}

	if `age' == 20 {
		save clean_data/dina_age.dta,replace
	}
	else {
		merge 1:1 year using clean_data/dina_age.dta, nogen
		save clean_data/dina_age.dta,replace
	}

}
}

********************************************************************************
* ------------------------------------------------------------------------------
* - NIPA/IMA data -
* ------------------------------------------------------------------------------
********************************************************************************

* ------------------------------------------------------------------------------
* - Load and format NIPA data -
* ------------------------------------------------------------------------------

import excel ${data_ima}/NIPA_Section2All_xls.xlsx, sheet("T20100-A") cellrange(A8:CO54) clear


* - Extract firm observation year -
local start_year = real(C[1])
di `start_year'

* - Select lines -
keep if inlist(A,"Line","3","9","12")|inlist(A,"13","14","15","26","30")

drop B C

sxpose,clear

* - Get names from lines -
foreach var of varlist * {
	local stub = `var'[1]
	di "`stub'"
	rename `var' nipa_`stub'
}

drop if _n == 1

* - Make numeric -
foreach var of varlist * {
	qui replace `var' = "" if `var' == "---"
	qui destring `var', replace
	replace `var' = 0 if missing(`var')
}

rename nipa_Line	year

save clean_data/imanipa_temp.dta,replace

* ------------------------------------------------------------------------------
* - Corporate Capital Share from NIPA tables -
* ------------------------------------------------------------------------------

import excel ${data_ima}/NIPA_Section1All_xls.xlsx, sheet("T11300-A") cellrange(A8:BV97) clear

keep if inlist(A,"Line","3","4","9")

drop B C

sxpose,clear

* - Get names from lines -
foreach var of varlist * {
	local stub = `var'[1]
	di "`stub'"
	rename `var' nipa1_`stub'
}

drop if _n == 1

foreach var of varlist * {
	destring `var', replace
}

rename nipa1_Line	year

tsset year

gen capital_share = 1 - nipa1_4/(nipa1_3 - nipa1_9)
gen capital_share_avg5 = (1/5)*(L2.capital_share + L.capital_share + capital_share + F.capital_share + F2.capital_share)


lab var capital_share "Corporate Sector Capital Share (NIPA)"

keep year capital_share*

merge 1:1 year using clean_data/imanipa_temp.dta, nogen keep(3)

save clean_data/imanipa_temp.dta,replace

* ------------------------------------------------------------------------------
* - Load and format IMA data -
* ------------------------------------------------------------------------------

import excel ${data_ima}/IMA_Section1All_xls.xlsx, sheet("IMAtS3.a-A") cellrange(A8:BJ157) clear

* - Select lines -
keep if inlist(A,"Line","19","99","103","104")|inlist(A,"109","114","117","121","123","131")

drop B C

sxpose,clear

* - Get lines from NIPA as variables names -
foreach var of varlist * {
	local stub = `var'[1]
	di "`stub'"
	rename `var' ima_`stub'
}

drop if _n == 1

* - Make numeric -
foreach var of varlist * {
	qui replace `var' = "" if `var' == "---" | `var' == "....."
	qui destring `var', replace
	replace `var' = 0 if missing(`var')
}

rename ima_Line year

merge 1:1 year using clean_data/imanipa_temp.dta, nogen keep(3)

save clean_data/imanipa_temp.dta,replace

* - PCE Price Deflator ---------------------------------------------------------

freduse DPCERG3A086NBEA,clear

rename DPCERG3A086NBEA deflator

* - Generate important variables -
gen year = real(substr(date,1,4))

keep year deflator

lab var deflator "PCE Index"

* - Merge with other variables -
merge 1:1 year using clean_data/imanipa_temp.dta, nogen keep(3)

* - Adjust date and value of IMA stock variables with deflator
foreach var of varlist ima* {
	replace `var' = `var' * (100/deflator)
	gen L`var' = L.`var'
	replace `var' = L`var' * (deflator/100)
	drop L`var'
}

drop deflator

save clean_data/imanipa_temp.dta,replace

* ------------------------------------------------------------------------------
* - Generate variables -
* ------------------------------------------------------------------------------

use clean_data/imanipa_temp.dta,clear

* - Income ---------------------------------------------------------------------

* - Individual items -
gen fkfix = nipa_14
gen fkequ = nipa_15
gen fkbus = nipa_9 * capital_share // Note from NIPA tables
gen fkhou = nipa_12 + (ima_19 - nipa_30)
gen fkdeb = ima_19 


* - Income aggregates -
gen imanipa_i_equ = fkequ + fkbus
gen imanipa_i_fass = fkequ + fkfix + fkbus 
gen imanipa_i_ass = imanipa_i_fass + fkhou

gen imanipa_i_fass_net = imanipa_i_fass + fkdeb
gen imanipa_i_ass_net = imanipa_i_ass + fkdeb

* - Stocks ---------------------------------------------------------------------
gen hwpen = ima_123
gen hwfix = ima_104 + ima_109 + ima_114 + ima_123
gen hwequ = ima_117 - ima_121
gen hwbus = ima_121
gen hwhou = ima_99
gen hwdeb = ima_131

* - Wealth aggregates -
gen imanipa_w_equ = hwequ + hwbus
gen imanipa_w_fass = hwequ + hwfix + hwbus 
gen imanipa_w_ass = imanipa_w_fass + hwhou
gen imanipa_w_pen = hwpen

gen imanipa_w_fass_net = imanipa_w_fass - hwdeb
gen imanipa_w_ass_net = imanipa_w_ass - hwdeb

* - Asset shares -
gen imanipa_sh_equ_fass = imanipa_w_equ/imanipa_w_fass_net*100
gen imanipa_sh_equ_fass_nopen = imanipa_w_equ/(imanipa_w_fass_net - imanipa_w_pen)*100
gen imanipa_sh_pen_fass = imanipa_w_pen/imanipa_w_fass_net*100

gen imanipa_sh_equ_ass = imanipa_w_equ/imanipa_w_ass_net*100
gen imanipa_sh_equ_ass_nopen = imanipa_w_equ/(imanipa_w_ass_net - imanipa_w_pen)*100

* - Counterfactual with constant pension share -
gen nonpension_share = (imanipa_w_fass_net - imanipa_w_pen)/imanipa_w_fass_net
qui sum nonpension_share if year == 1980
replace nonpension_share = `r(mean)'
gen imanipa_sh_equ_fass_copen = imanipa_w_equ/(imanipa_w_fass_net - imanipa_w_pen)* nonpension_share *100
drop nonpension_share


* - Income to wealth ratios -
gen imanipa_iwr_fass = imanipa_i_fass_net/imanipa_w_fass_net*100
gen imanipa_iwr_ass = imanipa_i_ass_net/imanipa_w_ass_net*100

keeporder year imanipa_sh_* imanipa_iwr_* 

save clean_data/imanipa_temp.dta,replace


********************************************************************************
* ------------------------------------------------------------------------------
* - DFA data -
* ------------------------------------------------------------------------------
********************************************************************************

* ------------------------------------------------------------------------------
* - Statistics by income percentiles -
* ------------------------------------------------------------------------------

* - Aggregate levels -----------------------------------------------------------

import delimited ${data_dfa}/dfa-other-socioeconomic-levels/dfa-income-levels.csv, clear

gcollapse (sum) assets-networth, by(date)

gen hwequ = corporateequitiesandmutualfundsh
gen hwbus = equityinnoncorpoatebusiness
gen hwpen = pensionentitlements + lifeinsurancereserves
gen hwfix = financialassets - hwequ - hwbus - hwpen
gen hwhou = realestate
gen hwdeb = liabilities

keep date hw*

save clean_data/dfa_temp.dta ,replace

* - Top 20% --------------------------------------------------------------------

import delimited ${data_dfa}/dfa-other-socioeconomic-levels/dfa-income-levels.csv, clear

keep if inlist(category,"pct99to100","pct80to99")

gcollapse (sum) assets-networth, by(date)

gen hwequ_qui = corporateequitiesandmutualfundsh
gen hwbus_qui = equityinnoncorpoatebusiness
gen hwpen_qui = pensionentitlements + lifeinsurancereserves
gen hwfix_qui = financialassets - hwequ - hwbus - hwpen
gen hwhou_qui = realestate
gen hwdeb_qui = liabilities

keep date hw*

merge 1:1 date using clean_data/dfa_temp.dta, keep(3) nogen

save clean_data/dfa_temp.dta ,replace

* - Top 1% --------------------------------------------------------------------

import delimited ${data_dfa}/dfa-other-socioeconomic-levels/dfa-income-levels.csv, clear

keep if inlist(category,"pct99to100")

gcollapse (sum) assets-networth, by(date)

gen hwequ_cen = corporateequitiesandmutualfundsh
gen hwbus_cen = equityinnoncorpoatebusiness
gen hwpen_cen = pensionentitlements + lifeinsurancereserves
gen hwfix_cen = financialassets - hwequ - hwbus - hwpen
gen hwhou_cen = realestate
gen hwdeb_cen = liabilities

keep date hw*

merge 1:1 date using clean_data/dfa_temp.dta, keep(3) nogen

save clean_data/dfa_temp.dta ,replace

* - Collect all data -----------------------------------------------------------

* - Annualize -
gen year = real(substr(date,1,4))

gcollapse(mean) hw* ,by(year)


* - Aggregates -
gen dfa_w_equ = hwequ + hwbus
gen dfa_w_fass = hwequ + hwfix + hwbus + hwpen
gen dfa_w_ass = dfa_w_fass + hwhou
gen dfa_w_pen = hwpen

gen dfa_w_fass_net = dfa_w_fass - hwdeb
gen dfa_w_ass_net = dfa_w_ass - hwdeb

gen dfa_sh_equ_fass = dfa_w_equ/dfa_w_fass_net*100
gen dfa_sh_equ_fass_nopen = dfa_w_equ/(dfa_w_fass_net-dfa_w_pen)*100
gen dfa_sh_pen_fass = dfa_w_pen/dfa_w_fass_net*100

gen dfa_sh_equ_ass = dfa_w_equ/dfa_w_ass_net*100
gen dfa_sh_equ_ass_nopen = dfa_w_equ/(dfa_w_ass_net-dfa_w_pen)*100

* - By income group -
foreach stub in cen qui {
	gen dfa_w_equ_`stub' = hwequ_`stub' + hwbus_`stub'
	gen dfa_w_fass_`stub' = hwequ_`stub' + hwfix_`stub' + hwbus_`stub' + hwpen_`stub'
	gen dfa_w_ass_`stub' = dfa_w_fass_`stub' + hwhou_`stub'
	gen dfa_w_pen_`stub' = hwpen_`stub'

	gen dfa_w_fass_net_`stub' = dfa_w_fass_`stub' - hwdeb_`stub'
	gen dfa_w_ass_net_`stub' = dfa_w_ass_`stub' - hwdeb_`stub'
	
	gen dfa_sh_equ_fass_`stub' = dfa_w_equ_`stub'/dfa_w_fass_net_`stub'*100
	gen dfa_sh_equ_fass_nopen_`stub' = dfa_w_equ_`stub'/(dfa_w_fass_net_`stub' - dfa_w_pen_`stub')*100
	gen dfa_sh_pen_fass_`stub' = dfa_w_pen_`stub'/dfa_w_fass_net_`stub'*100
	
	gen dfa_sh_equ_ass_`stub' = dfa_w_equ_`stub'/dfa_w_ass_net_`stub'*100
	gen dfa_sh_equ_ass_nopen_`stub' = dfa_w_equ_`stub'/(dfa_w_ass_net_`stub' - dfa_w_pen_`stub')*100
}


* - Counter-factual shares holding constant pension wealth share -
gen nonpension_share = (dfa_w_fass_net - dfa_w_pen)/dfa_w_fass_net
qui sum nonpension_share if year == 1990
replace nonpension_share = `r(mean)'
gen dfa_sh_equ_fass_copen = dfa_w_equ/(dfa_w_fass_net - dfa_w_pen)* nonpension_share *100
drop nonpension_share

foreach stub in cen qui {
	gen nonpension_share = (dfa_w_fass_net_`stub' - dfa_w_pen_`stub')/dfa_w_fass_net_`stub'
	qui sum nonpension_share if year == 1990
	replace nonpension_share = `r(mean)'
	gen dfa_sh_equ_fass_copen_`stub' = dfa_w_equ_`stub'/(dfa_w_fass_net_`stub' - dfa_w_pen_`stub')* nonpension_share *100
	drop nonpension_share

}

keeporder year dfa_sh_*

save clean_data/dfa_temp.dta,replace


********************************************************************************
* ------------------------------------------------------------------------------
* - Merge data -
* ------------------------------------------------------------------------------
********************************************************************************

use clean_data/dina_temp.dta,clear

merge 1:1 year using clean_data/dina_age.dta,nogen
merge 1:1 year using clean_data/imanipa_temp.dta, nogen
merge 1:1 year using clean_data/dfa_temp.dta, nogen

order year *_sh_* *_iwr_*

tsset year

save clean_data/dina_imanipa_dfa_data_v01.dta, replace
export delimited clean_data/dina_imanipa_dfa_data_v01.csv, replace

erase clean_data/imanipa_temp.dta
erase clean_data/dfa_temp.dta
erase clean_data/dina_temp.dta
erase clean_data/dina_age.dta
