% -------------------------------------------------------------------------
%
% Project: Uneven Growth
%
% Purpose: Plot figures of returns and shares invested in productive
% capital
%
% Version: 2 June 2022
% -------------------------------------------------------------------------

clear variables
close all

% change location to the folder where this m file is saved
mfile_name          = mfilename('fullpath');
[pathstr,name,ext]  = fileparts(mfile_name);
cd(pathstr);

% add all routines in the subfolders
cd ..
addpath 'clean_data'
addpath 'output'

% -------------------------------------------------------------------------
%% - Preamble -
% -------------------------------------------------------------------------
 
% Set latex as default interpreter
set(groot, 'defaulttextinterpreter','latex'); 
set(groot, 'defaultAxesTickLabelInterpreter','latex'); 
set(groot, 'defaultLegendInterpreter','latex');


% -------------------------------------------------------------------------
%% - Read data -
% -------------------------------------------------------------------------

D = readtable("kappa_shares.csv");

R = readtable("returns.csv");

% -------------------------------------------------------------------------
%% - Figure - Shares kappa --------------------
% -------------------------------------------------------------------------

Figure_kappa_I_shares = figure('units','centimeters','position',[10 10 7*(16/9)*2 10],'Color','w');

% Baseline
subplot(1,2,1)
hold on
h1=plot(D.year,D.dfa_sh_nw_KI_cen*100,  'r',  'LineWidth', 1.2);
h2=plot(D.year,D.dfa_sh_KI_cen*100,   'c',  'LineWidth', 1.2);
h3=plot(D.year,D.dina_sh_KI_cen*100,   'b',  'LineWidth', 1.2);
h4=plot(D.year,D.dina_sh_KI_dec*100,   'b--',  'LineWidth', 1.2);
h5=plot(D.year,D.dina_sh_KI_mil*100,   'b:',  'LineWidth', 1.2);

h7=xline(2007, '--', 'color', [0 0 0]+1/3 );
legend('DFA, top 1\% net worth', 'DFA, top 1\% income', 'DINA, top 1\% income', 'DINA, top 10\% income', 'DINA, top 0.1\% income', 'location', 'northwest');
legend('boxoff')
ylabel('$\kappa$, percent')
title('Including government bonds')
xlim([1960, 2020]) 
ylim([50 200])

% W/o govt bonds
subplot(1,2,2)
hold on
h1=plot(D.year,D.dfa_sh_nw_KI_no_safe_cen*100,  'r',  'LineWidth', 1.2);
h2=plot(D.year,D.dfa_sh_KI_no_safe_cen*100,   'c',  'LineWidth', 1.2);
h3=plot(D.year,D.dina_sh_KI_no_safe_cen*100,   'b',  'LineWidth', 1.2);
h4=plot(D.year,D.dina_sh_KI_no_safe_dec*100,   'b--',  'LineWidth', 1.2);
h5=plot(D.year,D.dina_sh_KI_no_safe_mil*100,   'b:',  'LineWidth', 1.2);
h6=plot(D.year,100+D.imanipa_sh_deb2equ*100,   'm',  'LineWidth', 1.2);

h7=xline(2007, '--', 'color', [0 0 0]+1/3 );
legend(h6, {'Additionally $D_I=L_H=L_I=0$ (NIPA/IMA)'}, 'location', 'northwest');
legend('boxoff')
ylabel('$\kappa$, percent')
title('Excluding government bonds, $B_I=0$')
xlim([1960, 2020])
ylim([50 200])


export_fig 'output/Figure_kappa_I_shares'.pdf

% ---------------------------------------------------------------------------------------------
%% - Figure - How much equity do we capture in different definitions of investor group 
% ---------------------------------------------------------------------------------------------

Figure_investors_equity = figure('units','centimeters','position',[10 10 7*(16/9) 8],'Color','w');

hold on
h1=plot(D.year,D.dfa_nw_equity_share_cen*100,  'r',  'LineWidth', 1.2);
h2=plot(D.year,D.dfa_equity_share_cen*100,   'c',  'LineWidth', 1.2);
h3=plot(D.year,D.dina_equity_share_cen*100,   'b',  'LineWidth', 1.2);
h4=plot(D.year,D.dina_equity_share_dec *100,   'b--',  'LineWidth', 1.2);
h5=plot(D.year,D.dina_equity_share_mil *100,   'b:',  'LineWidth', 1.2);

h7=xline(2007, '--', 'color', [0 0 0]+1/3 );
legend('DFA, top 1\% net worth', 'DFA, top 1\% income', 'DINA, top 1\% income', 'DINA, top 10\% income', 'DINA, top 0.1\% income', 'location', 'northwest');
legend('boxoff')
ylabel('$\frac{E_I}{E_I+E_H}$, percent')
title('Share of equity held by investors')
xlim([1960, 2020]) 
ylim([0 120])

export_fig 'output/Figure_investors_equity'.pdf

% -------------------------------------------------------------------------
%% - Figure - Returns --------------------
% -------------------------------------------------------------------------

Figure_returns = figure('units','centimeters','position',[10 10 7*(16/9)*2 10],'Color','w');

subplot(1,2,1)
hold on
h1=plot(R.year,R.rk_measure,  'k',  'LineWidth', 1.2);
h2=plot(R.year,R.r_star,  'r',  'LineWidth', 1.2);

h6=xline(2007, '--', 'color', [0 0 0]+1/3 );
legend('$r_K$', '$r_B$',  'location', 'northwest');
legend('boxoff')
ylabel('percent')
title('Safe and risky returns')
xlim([1960, 2020]) 
ylim([0,12])

subplot(1,2,2)
hold on
h1=plot(R.year,R.rw_dfa_nw_KI_cen,  'c',  'LineWidth', 1.2);
h2=plot(R.year,R.rw_dfa_nw_KI_no_safe_cen,  'c--',  'LineWidth', 1.2);
h3=plot(R.year,R.rw_dina_KI_cen,  'b',  'LineWidth', 1.2);
h4=plot(R.year,R.rw_dina_KI_no_safe_cen,  'b--',  'LineWidth', 1.2);
h5=plot(R.year,R.rw_benchmark,  'm',  'LineWidth', 1.2);
h6=xline(2007, '--', 'color', [0 0 0]+1/3 );
legend('$r_W$ (DFA)', '$r_W$ with $B_I=0$ (DFA)', '$r_W$ (DINA)', '$r_W$ with $B_I=0$ (DINA)', '$r_W$ with $B_I=D_I=L_I=0$',  'location', 'southwest');
legend('boxoff')
ylabel('percent')
title('Investors return')
xlim([1960, 2020]) 
ylim([0,12])

export_fig 'output/Figure_returns'.pdf

% -------------------------------------------------------------------------
%% - Figure - Returns - slides intro --------------------
% -------------------------------------------------------------------------

Figure_returns_slides_intro = figure('units','centimeters','position',[10 10 21 7],'Color','w');

subplot(1,2,1)
hold on
h1=plot(R.year,R.rk_measure-R.rk_measure(18),  'color', [0,0.5,0],  'LineWidth', 1.2);
h2=plot(R.year,R.rb_measure-R.rb_measure(18),  'r',  'LineWidth', 1.2);
%h6=xline(2007, '--', 'color', [0 0 0]+1/3 );
legend('Business capital', 'Safe assets',  'location', 'northwest');
legend('boxoff')
ylabel('percent')
title('After-tax returns')
xlim([1970, 2020]) 
ylim([-3,6])

subplot(1,2,2)
hold on

rk_adj2 = R.rk_measure - 2*R.grate_mean;
rb_adj2 = R.rb_measure - 2*R.grate_mean;

h1=plot(R.year,rk_adj2-rk_adj2(18),  'color', [0,0.5,0],  'LineWidth', 1.2);
h2=plot(R.year,rb_adj2-rb_adj2(18),  'r',  'LineWidth', 1.2);
%h6=xline(2007, '--', 'color', [0 0 0]+1/3 );
%legend('$r_K$', '$r_B$',  'location', 'northwest');
%legend('boxoff')
ylabel('percent')
title('After-tax returns - $\sigma \times g$')
xlim([1970, 2020]) 
ylim([-3,6])

export_fig 'output/Figure_returns_slides_intro'.pdf


% -------------------------------------------------------------------------
%% - Figure XX - Returns and Adjusted Returns --------------------
% -------------------------------------------------------------------------

Figure_returns_and_adj_returns_sigma2 = figure('units','centimeters','position',[10 10 7*(16/9)*2 10],'Color','w');

subplot(1,2,1)
hold on
h1=plot(R.year,R.rw_dfa_nw_KI_cen,  'r',  'LineWidth', 1.2);
h2=plot(R.year,R.rw_dfa_nw_KI_no_safe_cen,  'r--',  'LineWidth', 1.2);
h3=plot(R.year,R.rw_dina_KI_cen,  'b',  'LineWidth', 1.2);
h4=plot(R.year,R.rw_dina_KI_no_safe_cen,  'b--',  'LineWidth', 1.2);
h5=plot(R.year,R.rw_benchmark,  'k',  'LineWidth', 1.2);
h6=xline(2007, '--', 'color', [0 0 0]+1/3 );
legend('DFA (top 1\% net worth)', 'DFA with $B_I=0$', 'DINA (top 1\% income)', 'DINA with $B_I=0$ ', '$B_I=D_I=L_I=0$',  'location', 'southwest');
legend('boxoff')
ylabel('percent')
title('Return to wealth $r_W$')
xlim([1960, 2020]) 
ylim([-2,12])

subplot(1,2,2)
hold on
h1=plot(R.year,R.rw_dfa_nw_KI_cen_adj2,  'r',  'LineWidth', 1.2);
h2=plot(R.year,R.rw_dfa_nw_KI_no_safe_cen_adj2,  'r--',  'LineWidth', 1.2);
h3=plot(R.year,R.rw_dina_KI_cen_adj2,  'b',  'LineWidth', 1.2);
h4=plot(R.year,R.rw_dina_KI_no_safe_cen_adj2,  'b--',  'LineWidth', 1.2);
h5=plot(R.year,R.rw_benchmark_adj2,  'k',  'LineWidth', 1.2);
h6=xline(2007, '--', 'color', [0 0 0]+1/3 );
%legend('$r_W$ (DFA)', '$r_W$ with $B_I=0$ (DFA)', '$r_W$ (DINA)', '$r_W$ with $B_I=0$ (DINA)', '$r_W$ with $B_I=D_I=L_I=0$',  'location', 'southwest');
%legend('boxoff')
ylabel('percent')
title('Return gap $r_W-\sigma g$')
xlim([1960, 2020]) 
ylim([-2,12])


export_fig 'output/Figure_returns_and_adj_returns_sigma2'.pdf

% -------------------------------------------------------------------------
%% - Figure XX - Returns and Adjusted Returns 3 PANELS --------------------
% -------------------------------------------------------------------------
% 
% figure
% plot(R.year, R.grate_noadj, '--k')
% hold on
% plot(R.year, R.grate, 'k')
% plot(R.year, R.grate_noadj_mean, '--r')
% plot(R.year, R.grate_mean, 'r')
% legend('CBO real GDP per capita + fcast', 'Adjusted', '20 year MA', '20 year MA adj')
% 


Figure_returns_and_adj_returns_3panels = figure('units','centimeters','position',[10 10 7*(16/9)*2 10],'Color','w');

subplot(1,3,1)
hold on
h1=plot(R.year,R.rw_dfa_nw_KI_cen,  'r',  'LineWidth', 1.2);
h2=plot(R.year,R.rw_dfa_nw_KI_no_safe_cen,  'r--',  'LineWidth', 1.2);
h3=plot(R.year,R.rw_dina_KI_cen,  'b',  'LineWidth', 1.2);
h4=plot(R.year,R.rw_dina_KI_no_safe_cen,  'b--',  'LineWidth', 1.2);
h5=plot(R.year,R.rw_benchmark,  'k',  'LineWidth', 1.2);
h6=xline(2007, '--', 'color', [0 0 0]+1/3 );
legend('DFA (top 1\% net worth)', 'DFA with $B_I=0$', 'DINA (top 1\% income)', 'DINA with $B_I=0$ ', '$B_I=D_I=L_I=0$',  'location', 'southwest');
legend('boxoff')
ylabel('percent')
title('Return to wealth $\widehat{r}_W$')
xlim([1960, 2020]) 
ylim([-5,15])

subplot(1,3,2)
hold on
h1=plot(R.year,R.rw_dfa_nw_KI_cen_adj,  'r',  'LineWidth', 1.2);
h2=plot(R.year,R.rw_dfa_nw_KI_no_safe_cen_adj,  'r--',  'LineWidth', 1.2);
h3=plot(R.year,R.rw_dina_KI_cen_adj,  'b',  'LineWidth', 1.2);
h4=plot(R.year,R.rw_dina_KI_no_safe_cen_adj,  'b--',  'LineWidth', 1.2);
h5=plot(R.year,R.rw_benchmark_adj,  'k',  'LineWidth', 1.2);
h6=xline(2007, '--', 'color', [0 0 0]+1/3 );
%legend('$r_W$ (DFA)', '$r_W$ with $B_I=0$ (DFA)', '$r_W$ (DINA)', '$r_W$ with $B_I=0$ (DINA)', '$r_W$ with $B_I=D_I=L_I=0$',  'location', 'southwest');
%legend('boxoff')
ylabel('percent')
title('Return premium, $\sigma=1$')
xlim([1960, 2020]) 
ylim([-5,15])

subplot(1,3,3)
hold on
h1=plot(R.year,R.rw_dfa_nw_KI_cen_adj2,  'r',  'LineWidth', 1.2);
h2=plot(R.year,R.rw_dfa_nw_KI_no_safe_cen_adj2,  'r--',  'LineWidth', 1.2);
h3=plot(R.year,R.rw_dina_KI_cen_adj2,  'b',  'LineWidth', 1.2);
h4=plot(R.year,R.rw_dina_KI_no_safe_cen_adj2,  'b--',  'LineWidth', 1.2);
h5=plot(R.year,R.rw_benchmark_adj2,  'k',  'LineWidth', 1.2);
h6=xline(2007, '--', 'color', [0 0 0]+1/3 );
%legend('$r_W$ (DFA)', '$r_W$ with $B_I=0$ (DFA)', '$r_W$ (DINA)', '$r_W$ with $B_I=0$ (DINA)', '$r_W$ with $B_I=D_I=L_I=0$',  'location', 'southwest');
%legend('boxoff')
ylabel('percent')
title('Return premium, $\sigma=2$')
xlim([1960, 2020]) 
ylim([-5,15])


export_fig 'output/Figure_returns_and_adj_returns_3panels'.pdf

% -------------------------------------------------------------------------
%% - Figure XX - Returns and Adjusted Returns for slides --------------------
% -------------------------------------------------------------------------

Figure_returns_and_adj_returns_slides = figure('units','centimeters','position',[10 10 22 7],'Color','w');

subplot(1,2,1)
hold on
h1=plot(R.year,R.rw_dfa_nw_KI_cen,  'r',  'LineWidth', 1.2);
h2=plot(R.year,R.rw_dfa_nw_KI_no_safe_cen,  'r--',  'LineWidth', 1.2);
h3=plot(R.year,R.rw_dina_KI_cen,  'b',  'LineWidth', 1.2);
h4=plot(R.year,R.rw_dina_KI_no_safe_cen,  'b--',  'LineWidth', 1.2);
h5=plot(R.year,R.rw_benchmark,  'k',  'LineWidth', 1.2);
h6=xline(2007, '--', 'color', [0 0 0]+1/3 );
legend('DFA (top 1\% net worth)', 'DFA with $B_I=0$', 'DINA (top 1\% income)', 'DINA with $B_I=0$ ', '$B_I=D_I=L_I=0$',  'location', 'southwest');
legend('boxoff')
ylabel('percent')
title('Return to wealth $r_W$')
xlim([1960, 2020]) 
ylim([0,12])

subplot(1,2,2)
hold on
h1=plot(R.year,R.rw_dfa_nw_KI_cen_adj,  'r',  'LineWidth', 1.2);
h2=plot(R.year,R.rw_dfa_nw_KI_no_safe_cen_adj,  'r--',  'LineWidth', 1.2);
h3=plot(R.year,R.rw_dina_KI_cen_adj,  'b',  'LineWidth', 1.2);
h4=plot(R.year,R.rw_dina_KI_no_safe_cen_adj,  'b--',  'LineWidth', 1.2);
h5=plot(R.year,R.rw_benchmark_adj,  'k',  'LineWidth', 1.2);
h6=xline(2007, '--', 'color', [0 0 0]+1/3 );
%legend('$r_W$ (DFA)', '$r_W$ with $B_I=0$ (DFA)', '$r_W$ (DINA)', '$r_W$ with $B_I=0$ (DINA)', '$r_W$ with $B_I=D_I=L_I=0$',  'location', 'southwest');
%legend('boxoff')
ylabel('percent')
title('Return premium, $\sigma=1$')
xlim([1960, 2020]) 
ylim([0,12])


export_fig 'output/Figure_returns_and_adj_returns_slides'.pdf


% -------------------------------------------------------------------------
%% - Figure XX - Returns to different assets in the US --------------------
% -------------------------------------------------------------------------

Figure_paper_returns = figure('units','centimeters','position',[10 10 12*(15/9) 10],'Color','w');

hold on
plot(R.year,R.rmean_ckp_bv_cc, 'color', [0,0.5,0],'LineWidth',1.2);
message = sprintf('Return to productive capital \nin the non-financial corporate \nsector (IMA)');
text(1967,1.05*R.rmean_ckp_bv_cc(25)+1.6,message, 'color', [0,0.5,0]);

plot(R.year,R.rmean_pk_bv_hh, 'r-', 'LineWidth',1.2);
message = sprintf('Return to productive capital \nin the non-corporate \nsector (IMA/NIPA)');
text(2013.5,R.rmean_pk_bv_hh(56)+1.3,message, 'color', 'r');

plot(R.year,R.grr_rk_bs_post_avgrev,'k-', 'LineWidth',1.2);
message = sprintf('Return to business capital \n(GRR)');
text(2009,R.grr_rk_bs_post_avgrev(32)-0.2,message,'color', 'k');

plot(R.year,R.r_star,'b-' , 'LineWidth',1.2);
message = sprintf('Estimated $r^*$  \n (HLW)');
text(2020,R.r_star(56)+0.5,message,'color', 'b');

ylabel('annual returns, percent')
%title('Return measures for US capital')
xlim([1960, 2029])
%ylim([-1 11])
set(gca,'ytick',0:2:10)
set(gcf, 'Color', 'w');
  
export_fig 'output/Figure_paper_returns'.pdf


% -------------------------------------------------------------------------
%% - Figure XX - Returns and kappa_I (slides)  --------------------
% -------------------------------------------------------------------------

figure_17 = figure('units','centimeters','position',[10 10 25 8],'Color','w');


subplot(1,2,1)
hold on
% plot(R.year,R.ret_corp, 'k--','LineWidth',0.8);
% message = sprintf('HH side: corporate');
% text(1983,0.72*R.ret_corp(56),message, 'color', 'k');
% 
% plot(R.year,R.ret_noncorp, 'k:', 'LineWidth',1);
% message = sprintf('HH side: noncorporate');
% text(1970,R.ret_noncorp(56),message, 'color', 'k');

% plot(R.year,R.grr_rk_bs_post_avgrev,'k-', 'LineWidth',0.8);
% message = sprintf('Firm side \n(Gomme et al)');
% text(2018,R.grr_rk_bs_post_avgrev(56),message,'color', 'k');


plot(R.year,R.rmean_ckp_bv_cc, 'color', [0,0.5,0],'LineWidth',1.2);
message = sprintf('Productive capital \nin the non-financial corporate sector');
text(1967,1.11*R.rmean_ckp_bv_cc(25)+1.6,message, 'color', [0,0.5,0]);

plot(R.year,R.rmean_pk_bv_hh, 'r-', 'LineWidth',1.2);
message = sprintf('Privately held \nproductive capital');
text(2019,R.rmean_pk_bv_hh(56)+0.3,message, 'color', 'r');

plot(R.year,R.grr_rk_bs_post_avgrev,'k-', 'LineWidth',1.2);
message = sprintf('Business capital \n(GRR)');
text(2009,R.grr_rk_bs_post_avgrev(32)-0.2,message,'color', 'k');

plot(R.year,R.r_star,'b-' , 'LineWidth',1.2);
message = sprintf('Estimated $r_B$  \n (HLW)');
text(2020,R.r_star(56)+0.5,message,'color', 'b');
% 
% plot(R.year,R.r_star,'b-' , 'LineWidth',2);
% message = sprintf('$r_B$');
% text(2021,R.r_star(56),message,'color', 'b', 'Fontsize', 14);
% 
% plot(R.year,R.rk_measure,'k' , 'LineWidth',2);
% message = sprintf('$r_K$');
% text(2019,R.rk_measure(56),message,'color', 'k', 'Fontsize', 14);

ylabel('annual returns, percent')
title('Estimates of $r_K$ and $r_B$')
xlim([1960, 2029])
ylim([-1 11])
set(gca,'ytick',0:2:10)
set(gcf, 'Color', 'w');

subplot(1,2,2)
hold on
h1=plot(D.year,D.dfa_sh_nw_KI_no_safe_cen*100,  'r',  'LineWidth', 1.2);
h2=plot(D.year,D.dfa_sh_KI_no_safe_cen*100,   'r--',  'LineWidth', 1.2);
h3=plot(D.year,D.dina_sh_KI_no_safe_cen*100,   'b',  'LineWidth', 1.2);
h4=plot(D.year,D.dina_sh_KI_no_safe_dec*100,   'b--',  'LineWidth', 1.2);
h5=plot(D.year,D.dina_sh_KI_no_safe_mil*100,   'b:',  'LineWidth', 1.2);
h6=plot(D.year,100+D.imanipa_sh_deb2equ*100,   'k',  'LineWidth', 1.2);
%h7=xline(2007, '--', 'color', [0 0 0]+1/3 );
legend('DFA, top 1\% net worth', 'DFA, top 1\% income', 'DINA, top 1\% income', 'DINA, top 10\% income', 'DINA, top 0.1\% income', 'Equity-only investor', 'location', 'northwest');
legend('boxoff')
ylabel('share, percent')
title('Estimates of $\kappa$')
xlim([1960, 2020])
ylim([100 200])

  
export_fig 'output/Figure_slides_returns'.pdf



% -------------------------------------------------------------------------
%% - Figure XX -Components of returns--------------------------------------
% -------------------------------------------------------------------------

figure_components = figure('units','centimeters','position',[10 10 7*(16/9)*2 10],'Color','w');
 
% - Panel.A Income based -
subplot(1,2,1)

hold on



hold on
plot(R.year,R.rinc_ckp_bv_cc,'color', [0,0.5,0], 'LineWidth',1.2);
message = sprintf('Productive capital \nin the non-financial \ncorporate sector \n(IMA)');
text(2015,R.rinc_ckp_bv_cc(56)+1.5,message, 'color', [0,0.5,0]);

plot(R.year,R.rinc_pk_bv_hh, 'r', 'LineWidth',1.2);
message = sprintf('Productive capital \nin the non-corporate \nsector (IMA/NIPA)');
text(2010,R.rinc_pk_bv_hh(56)-1.5,message, 'color', 'r');

plot(R.year,R.grr_rk_bs_post_noc,'k', 'LineWidth', 1.2);
message = sprintf('Business \ncapital \n(GRR)');
text(2020,R.grr_rk_bs_post_noc(56),message,'color', 'k');

ylabel('annual income returns, percent')
title('Panel A. income component')

xlim([1960, 2030])
ylim([0,10])
set(gca,'ytick',0:1:30,'xtick',1960:10:2020)

 
% - Panel.B Revaluations - 
subplot(1,2,2)
hold on
plot(R.year,R.rev_ckp_bv_cc, 'color', [0,0.5,0], 'LineWidth',1);
message = sprintf('Productive capital \nin the non-financial \ncorporate sector \n(IMA)');
text(2020,3,message, 'color', [0,0.5,0]);

plot(R.year,R.rev_pk_bv_hh, 'r', 'LineWidth',1);
message = sprintf('Productive capital \nin the non-corporate \nsector (IMA/NIPA)');
text(2007,13,message, 'color',  'r');

plot(R.year,R.grr_rk_bs_post_rev, 'k', 'LineWidth',1.2);
message = sprintf('Business \ncapital \n(GRR)');
text(2020,R.grr_rk_bs_post_rev(56)-3,message,'color', 'k');


ylabel('annual revaluations, percent')
title('Panel B. revaluation component')
xlim([1960, 2030])
set(gca,'ytick',-25:5:50,'xtick',1960:10:2020)
 
export_fig output/Figure_paper_returns_composition.pdf


% -------------------------------------------------------------------------
%% - Figure XX -Return on equity--------------------------------------
% -------------------------------------------------------------------------

figure_return_on_equity = figure('units','centimeters','position',[10 10 7*(16/9)*2 10],'Color','w');


subplot(1,3,1)
hold on
plot(R.year,R.rinc_eq_mv_cc,  'b',  'LineWidth', 1.2);
plot(R.year,R.rinc_eq_mv_hh,  'r',  'LineWidth', 1.2);
plot(R.year,R.rinc_ave_hh, 'k',  'LineWidth', 1.2);
xline(2007, '--', 'color', [0 0 0]+1/3 );
legend('US non-financial corporate', 'Households: corporate equity', 'Households: average equity',   'location', 'southwest');
legend('boxoff')
ylabel('percent')
title('Dividend yield')
xlim([1960, 2020]) 
ylim([0,10])

subplot(1,3,2)
hold on
h1=plot(R.year,R.rexp_eq_mv_hh,  'r',  'LineWidth', 1.2);
h2=plot(R.year,R.rexp_eq_mv_cc,  'b',  'LineWidth', 1.2);
h3=plot(R.year,R.rexp_ave_hh, 'k',  'LineWidth', 1.2);
h4=xline(2007, '--', 'color', [0 0 0]+1/3 );
%legend('Household perspective', 'Corporate perspective',  'location', 'southwest');
%legend('boxoff')
ylabel('percent')
title('With 10-yr-forward average revaluations')
xlim([1960, 2020]) 
ylim([0,25])

subplot(1,3,3)
hold on
h1=plot(R.year,R.rreal_eq_mv_hh,  'r',  'LineWidth', 1.2);
h2=plot(R.year,R.rreal_eq_mv_cc,  'b',  'LineWidth', 1.2);
h3=plot(R.year,R.rreal_ave_hh,  'k',  'LineWidth', 1.2);
h4=xline(2007, '--', 'color', [0 0 0]+1/3 );
%legend('Household perspective', 'Corporate perspective',  'location', 'southwest');
%legend('boxoff')
ylabel('percent')
title('With actual revaluations')
xlim([1960, 2020]) 
ylim([-30,40])


export_fig 'output/Figure_returns_equity'.pdf



