% -------------------------------------------------------------------------
%
% Project: Uneven Growth
%
% Purpose: Function with all conditions that need to hold
%
% -------------------------------------------------------------------------

function [Difference] = equations_v09(Guess,Parameter)
    
    % - Extract parameters ------------------------------------------------

    lambda      = Parameter.lambda;          % Lambda (Credit constraint)
    gamnu       = Parameter.gamnu;           % nu x sigma^2
    rho         = Parameter.rho;             % Rho (Time preference)
    sigma       = Parameter.sigma;           % Sigma (Intertemporal Elasticity of Substitution)
    g           = Parameter.g;               % Economy wide growth rate
    chi         = Parameter.chi;             % Share of Investors
    p           = Parameter.p;               % Death probability 
    alpha       = Parameter.alpha;           % Capital share
    govbonds    = Parameter.govbonds;        % Total bonds
    delta       = Parameter.delta;           % Depreciation rate

    % - Extract guesses for values -
    k   = exp(Guess(1));                 % Capital stock (normalized by wages)
    b_h = Guess(2);                      % Bond holdings by households (normalized by wages)
    r_b = g+exp(Guess(3));               % Return to bonds
    
    % - Calculate useful variables -
    r_k = (alpha/(1-alpha))*(r_b-g)/k - delta;      % From equation 7 in the notes
    kap = max(0,min(1/lambda,(r_k-r_b)/gamnu));    % Optimal portfolio choice in notes
    x_e = k + (govbonds - b_h) + chi;               % normalized effective wealth of Investors
    x_h = b_h + (1-chi);                            % normalized effective wealth of households
    r_bar = kap * r_k + (1-kap) * r_b;              % return for investors
    
    % - Calculate differences across equations -
    Difference      = zeros(3,1);
    Difference(1)   = (1/sigma)*(r_bar - rho - g*sigma + 0.5 * (sigma - 1) * kap^2 * gamnu) * x_e - p * (k + (govbonds - b_h));  % eq 4 in notes
    Difference(2)   = kap*x_e - k;  % eq 5 in notes
    Difference(3)   = (1/sigma)*(r_b - rho - sigma*g) * x_h - p * b_h; % eq 6 in notes
    
end