% equilibrium.m
% This program implements the algorithm for computing decentralized
% equilibria, described in Appendix C of
% "Knowledge Growth and the Allocation of Time"
% It computes the triple of functions (v,phi,sigma) and a number gamma
% satisfying equations (9), (10) and (12) in the paper

clear all
close all
clc

%PARAMETERS
r = 0.06; %discount rate, rho
the = 0.5; %inverse of tail parameter of initial distribution
k = 0.05; %level parameter of tail of initial distribution

%MEETING TECHNOLOGY: alpha(s) = al0*s^eta;
eta = 0.3;
al0 = 0.0849; %chosen to deliver gamma=0.02

%GRID
I =2001;
xmin = 0;
xmax = 3;

xmin1 = 10^(-5);
logx = linspace(log(xmin1),log(xmax),I-1)';
x = exp(logx);
x = [xmin;x];

h = zeros(I,1);
for i=2:I
    h(i)=x(i)-x(i-1);
end


v = zeros(I,1);
f = zeros(I,1);
F = zeros(I,1);
psi = zeros(I,1);
al = zeros(I,1);

%ITERATION LIMITS
maxit = 20; %MAIN ITERATION
maxit_g = 100; %VALUE FUNCTION ITERATION
maxit_v = 100; %VALUE FUNCTION ITERATION

crit = .0000001; % criteria for convergence
crit_v = .0000001; % criteria for convergence
crit_g = .0000001; % criteria for convergence

% initialization of the iteration
g = 0.02;
v0 = x/(r-g); %Initial guess for value function
F0 = exp(-k.*x.^(-1/the)); %initial guess for BGP CDF
f0 = (k/the).*x.^(-1/the-1).*exp(-k.*x.^(-1/the)); %initial guess for BGP density
f0(1)=0;


relax=0.5;


%%%%%%%%%%%%%%%%%%%%%%%%
% START ITERATION ON f %
%%%%%%%%%%%%%%%%%%%%%%%%
for it=1:maxit

disp('it =')
disp(it)


%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% LOOP ON BELLMAN EQUATION %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%
S = zeros(I,1);

for it_v =1:maxit_v; %START VALUE FUNCTION ITERATION
for i=1:I
    diff = v0 - v0(i);
    S(i) = diff(i:I)'*(f0(i:I).*h(i:I));
end
sigu = ((al0*eta.*S)./x).^(1/(1-eta));
sig = max(min(sigu,1),0);
al = al0*sig.^eta;
%CREATE MATRICES
CC = al*(f0.*h)';
C = triu(CC,0);

B = zeros(I,I);
for i=2:I
    B(i,i) = r - g + g*x(i)/h(i) + al(i)*(1-F0(i));
    B(i,i-1) = - g*x(i)/h(i);
end
B(1,1) = r - g + al(1)*(1-F0(1));
A = B - C;
vec = (1-sig).*x;

v = A\vec; %solve system of I equations in I unknowns

vdist = max(max(abs(v0-v)));
if vdist <crit_v
break
end

v0=v; %update value function

end

if it_v == maxit_v
    disp('CAREFUL: ITERATION LIMIT ON VALUE FUNCTION ITERATION')
end


%%%%%%%%%%%%%%%%
% DISTRIBUTION %
%%%%%%%%%%%%%%%%

for it_g = 1:maxit_g
psi(I)=g/the;
F(I)=1-k*x(I)^(-1/the);
f(I)=(k/the)*x(I)^(-1/the-1);
for i=I:-1:2
    f(i-1)= h(i)/(g*x(i))*(f(i)*psi(i) - al(i)*f(i)*(1-F(i)) + f(i)*g) + f(i);
    F(i-1) = F(i) - f(i)*h(i);
    psi(i-1) = psi(i) - h(i)*al(i)*f(i);
end

%%%%%%%%%%%%%%%%%%%%%%
% UPDATE GROWTH RATE %
%%%%%%%%%%%%%%%%%%%%%%
gnew(it_g) = the*(al'*(f.*h) + al(I)*(1-F(I)) + al(1)*F(1));
gnew(it_g) = relax*gnew(it_g) + (1-relax)*g;
gnew(it_g)=min(the*max(al),gnew(it_g));
dist_g = max(abs(g-gnew(it_g)));
if dist_g <crit_g
break
end
g = gnew(it_g);
end

if it_g == maxit_g
    disp('CAREFUL: ITERATION LIMIT ON GAMMA ITERATION')
end

dist = max(abs(f-f0));
f0_it(:,it) = f0;
f_it(:,it) = f;
dist_it(it)=dist;
if dist <crit
break
end

f0=f;
F0=F;

end
if it == maxit
    disp('CAREFUL: ITERATION LIMIT ON MAIN ITERATION')
end

save equilibrium.mat al sig x I h v g f F r the al0 k eta